#!/bin/bash

SCRIPT_PATH="$( cd -- "$(dirname "$0")" >/dev/null 2>&1 ; pwd -P )"
LOG_DIR="/opt/rdi/logs"
K3S_UNINSTALL_SCRIPT="/usr/local/bin/k3s-uninstall.sh"

die() {
    local msg="$1"
    echo "FATAL: ${msg}" >&2
    exit 1
}

check_root() {
    [ "$EUID" -ne 0 ] && die "This script must be run as root or via sudo" >&2
}


is_valid_ubuntu() {
    [ -f /etc/os-release ] && grep -q "Ubuntu \(20\|22\|24\)" /etc/os-release
}

is_rhel_8() {
    [ -f /etc/redhat-release ] && grep -q "release 8" /etc/redhat-release
}

is_rhel_9() {
    [ -f /etc/redhat-release ] && grep -q "release 9" /etc/redhat-release
}

get_rdi_cli_dir() {
    if is_valid_ubuntu; then
        echo "ubuntu-20.04"
    elif is_rhel_8 || is_rhel_9; then
        echo "rhel-8"
    else
        die "Unsupported Linux distribution or version. Supported distributions: Ubuntu (20, 22, 24), RHEL 8, RHEL 9" >&2
    fi
}

setup_sudo_env() {
    # When running with sudo, set HOME to the sudo user's home directory.
    # This ensures that files (like .redis-di) are created in the correct location
    # rather than in root's home directory.
    if [ -n "${SUDO_USER}" ]; then
        HOME=$(getent passwd ${SUDO_USER} | cut -d: -f6)
        export HOME
    fi
}

setup_binary() {
    local redis_di_path="$1"
    [ ! -e "${redis_di_path}" ] && die "File not found: ${redis_di_path}"

    cp -f "${redis_di_path}" /usr/local/bin
    ln -sf /usr/local/bin/redis-di /usr/bin/redis-di
}

setup_logs() {
    if [ ! -d "${LOG_DIR}" ]; then
        mkdir -p "${LOG_DIR}"
        chmod 777 "${LOG_DIR}"
    fi
}

check_user_confirmation() {
    local prompt="$1"
    local default="${2:-N}"

    while true; do
        read -rp "${prompt} [y, N] " user_input
        user_input=${user_input:-$default}

        case "${user_input}" in
            [yY])
                echo "Proceeding..."
                return 0
                ;;
            [nN])
                echo "Operation canceled."
                exit 0
                ;;
            *)
                echo "Invalid input. Please enter 'y' or 'N'."
                ;;
        esac
    done
}
